# Copyright (C) 2018, 2019, 2020, 2021 The Meme Factory, Inc.
# http://www.karlpinc.com/

# This file is part of PGWUI_Upload_Core.
#
# This program is free software: you can redistribute it and/or
# modify it under the terms of the GNU Affero General Public License
# as published by the Free Software Foundation, either version 3 of
# the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License along with this program.  If not, see
# <http://www.gnu.org/licenses/>.
#

# Karl O. Pinc <kop@karlpinc.com>

import markupsafe
import pytest
from pyramid.testing import DummyRequest
from pgwui_common.__init__ import includeme as pgwui_common_includeme
from pgwui_core import constants
from pgwui_upload_core.__init__ import includeme as pgwui_upload_core_includeme
from pgwui_upload_core import exceptions as upload_ex
from pgwui_upload_core.views import upload
from pgwui_develop import testing

# Activiate our pytest plugin
pytest_plugins = ("pgwui",)

# Mark all tests with "unittest"
pytestmark = pytest.mark.unittest

# Constants
CHANGED_RESPONSE = {
    'db': 'somedb',
    'db_changed': True,
    'filename': 'file',
    'lines': 5,
    'null_rep': 'NULL',
    'table': 'sometable',
    'trim_upload': constants.CHECKED,
    'upload_null': constants.CHECKED,
    'user': 'someuser',
}

UNCHANGED_RESPONSE = {'db_changed': False}

# A "upload form"
UPLOAD_FORM_W_LIT_CHECKED = {'literal_col_headings': True}

HOME_PAGE_SETTINGS = {'type': 'URL',
                      'source': '/'}

DEFAULT_URLS = {'pgwui_upload': '/upload',
                'pgwui_logout': '/logout',
                'home_page': '/'}


mock_escape = testing.make_mock_fixture(
    markupsafe, 'escape')


# Helper classes

class MockUploadEngine():
    def __init__(self, run_result):
        self.run_result = run_result

    def run(self):
        return self.run_result


class MockTableUploadHandler():
    def __init__(self, *args):
        pass


# Fixtures

@pytest.fixture
def isolate_upload_view(monkeypatch, pyramid_request_config):
    '''Keep upload_view() from calling anything

    Also, have isolate_upload_view(response)
    cause UploadEngine to return the supplied "response".
    '''
    def run(response):
        def upload_engine(*args):
            return MockUploadEngine(response)

        monkeypatch.setattr(upload, 'UploadEngine', upload_engine)
        monkeypatch.setattr(
            upload, 'BaseTableUploadHandler', MockTableUploadHandler)

        settings = pyramid_request_config.get_settings()
        settings['pgwui'] = settings.get('pgwui', dict())
        settings['pgwui'].update({'home_page': HOME_PAGE_SETTINGS})
        pgwui_common_includeme(pyramid_request_config)
        pgwui_upload_core_includeme(pyramid_request_config)
        settings['pgwui'].update({'urls': DEFAULT_URLS})
        pyramid_request_config.add_settings(settings)

    return run


# Tests

# BaseTableUploadHandler()

# BaseTableUploadHandler.__init__()

mock_tuh_init = testing.instance_method_mock_fixture('__init__')


# BaseTableUploadHandler.resolve_table()

mock_resolve_table = testing.instance_method_mock_fixture('resolve_table')


# BaseTableUploadHandler.good_table()

mock_good_table = testing.instance_method_mock_fixture('good_table')


@pytest.fixture
def neuter_tableuploadhandler(monkeypatch):
    '''Make TableUploadHander have the given uploadform
    '''
    def run(uploadform, request):
        uh = upload.BaseTableUploadHandler(request)
        monkeypatch.setattr(uh, 'uf', uploadform)

        return uh

    return run


# BaseTableUploadHandler.quote_columns()

@pytest.fixture
def get_quote_columns(neuter_tableuploadhandler):
    '''Return result of the upload handler's get_form_column_quoter()
    '''
    def run(uploadform, settings):
        request = DummyRequest()
        request.registry.settings = settings

        uh = neuter_tableuploadhandler(uploadform, DummyRequest())
        return uh.quote_columns(settings)

    return run


@pytest.mark.parametrize(
    ('val', 'expected'), [
        ('yes-always', True),
        ('choice-yes', True),
        ('choice-no', False),
        ('no-never', False)])
def test_tableuploadhandler_quote_columns(get_quote_columns, val, expected):
    '''When the settings ask for literal_column_headings = on return
    True
    '''
    result = get_quote_columns(UPLOAD_FORM_W_LIT_CHECKED,
                               {'literal_column_headings': val})
    assert result is expected


# BaseTableUploadHandler.validate_table()

def test_validate_table_good(
        mock_resolve_table, mock_good_table, mock_escape):
    '''When the table is good, the results of resolve_table() are returned
    '''
    expected = ('schema', 'table')

    request = DummyRequest()
    uh = upload.BaseTableUploadHandler(request)
    mocked_resolve_table = mock_resolve_table(uh)
    mocked_good_table = mock_good_table(uh)

    mocked_resolve_table.return_value = expected
    mocked_good_table.return_value = True
    result = uh.validate_table(None)

    assert result == expected


def test_validate_table_bad(
        mock_resolve_table, mock_good_table, mock_escape):
    '''When the table is not good, the right exception is raised
    '''
    expected = ('schema', 'table')

    request = DummyRequest()
    uh = upload.BaseTableUploadHandler(request)
    mocked_resolve_table = mock_resolve_table(uh)
    mocked_good_table = mock_good_table(uh)

    mocked_resolve_table.return_value = expected
    mocked_good_table.return_value = False
    with pytest.raises(upload_ex.CannotInsertError):
        uh.validate_table(None)

    assert True


# BaseTableUploadHandler.report_bad_cols()

@pytest.mark.parametrize(
    ('quotecols',), [
        (True,),
        (False,)])
def test_report_bad_cols(mock_escape, quotecols):
    '''The expected exception is raised
    '''

    request = DummyRequest()
    uh = upload.BaseTableUploadHandler(request)
    with pytest.raises(upload_ex.BadHeadersError):
        uh.report_bad_cols(None, ['col1', 'col2'], quotecols)

    assert True
