# Copyright (C) 2020, 2021 The Meme Factory, Inc.  http://www.karlpinc.com/

# This file is part of PGWUI_Menu.
#
# This program is free software: you can redistribute it and/or
# modify it under the terms of the GNU Affero General Public License
# as published by the Free Software Foundation, either version 3 of
# the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public
# License along with this program.  If not, see
# <http://www.gnu.org/licenses/>.
#

# Karl O. Pinc <kop@karlpinc.com>

import pytest

import pgwui_menu.check_settings as check_settings

from pgwui_common import checkset
from pgwui_develop import testing
from pgwui_menu import exceptions as menu_ex

# Activiate our pytest plugin
pytest_plugins = ("pgwui",)


# Module packaging test

def test_check_setting_is_pgwui_check_settings(
        pgwui_check_settings_entry_point):
    '''Ensure that pgwui_menu has a pgwui.check_settings entry point
    '''
    assert (pgwui_check_settings_entry_point('pgwui_menu.check_settings')
            is True)


# Mocks

mock_unknown_settings = testing.make_mock_fixture(
    checkset, 'unknown_settings')

mock_require_settings = testing.make_mock_fixture(
    checkset, 'require_settings')

mock_boolean_settings = testing.make_mock_fixture(
    checkset, 'boolean_settings')


# validate_order

@pytest.mark.unittest
def test_validate_order_nosetting():
    '''No error is delivered when there's no setting'''
    errors = []
    check_settings.validate_order(errors, None, {})

    assert errors == []


@pytest.mark.unittest
def test_validate_order_ok():
    '''No errors when all components in the ordering are ok
    '''
    errors = []
    check_settings.validate_order(
        errors, ['comp1', 'comp2'], {'order': ['comp1', 'comp2']})

    assert errors == []


@pytest.mark.unittest
def test_validate_order_singleton():
    '''A non-list as an ordering gets the right error
    '''
    errors = []
    check_settings.validate_order(
        errors, None, {'order': 'string'})

    assert errors
    assert isinstance(errors[0], menu_ex.BadOrderValuesError)


@pytest.mark.unittest
def test_validate_order_bad_component():
    '''Deliver error when a bad component name is supplied in the ordering
    '''
    errors = []
    check_settings.validate_order(
        errors, ['comp1', 'comp2'], {'order': ['bad_component']})

    assert errors
    assert isinstance(errors[0], menu_ex.BadOrderItemError)


order_err = 'order error'
mock_validate_order = testing.make_mock_fixture(
    check_settings, 'validate_order',
    wraps=lambda errors, *args: errors.append(order_err))


# check_settings()

@pytest.mark.unittest
def test_check_settings(mock_unknown_settings,
                        mock_require_settings,
                        mock_boolean_settings,
                        mock_validate_order):
    '''The setting checking functions are called once, the check_settings()
    call returns all the errors from each mock.
    '''

    unknown_retval = ['unk err']
    require_retval = ['req err']
    boolean_retval = ['bool err']

    mock_unknown_settings.return_value = unknown_retval
    mock_require_settings.return_value = require_retval
    mock_boolean_settings.return_value = boolean_retval

    result = check_settings.check_settings({})

    mock_unknown_settings.assert_called_once
    mock_require_settings.assert_called_once
    mock_boolean_settings.assert_called_once
    mock_validate_order.assert_called_once

    assert result.sort() == ([order_err]
                             + unknown_retval
                             + require_retval
                             + boolean_retval).sort()
