# Copyright (C) 2017 The Meme Factory, Inc.  http://www.meme.com/
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# Karl O. Pinc <kop@meme.com>

from enforcer import printout


# Constants

EXAMPLE_PATH = 'some/path'


#
# Class tests
#

# Helper classes

class MockDirobject(object):
    def __init__(self):
        self.path = EXAMPLE_PATH


# Printer

# print_section()

def test_printer_print_section_first(capsys):
    '''Does the first print section begin and end with break lines?'''

    printer = printout.Printer()
    printer.print_section('line of text\n')
    (out, err) = capsys.readouterr()
    lines = out.split('\n')
    assert lines[0] == printout.BREAK
    assert lines[-2] == printout.BREAK  # -2 because output ends in a \n


def test_printer_print_section_last(capsys):
    '''Printing 2 sections should result in 3 breaks'''

    printer = printout.Printer()
    printer.print_section('line of text\n')
    printer.print_section('line in a new section\n')
    (out, err) = capsys.readouterr()
    lines = out.split('\n')
    assert lines.count(printout.BREAK) == 3


# count()

def test_printer_counts():
    '''Printing a section bumps the item counter'''
    printer = printout.Printer()
    count = printer.count
    printer.print_section('line of text\n')
    assert printer.count == count + 1


# add_report_items()

def test_printer_add_report_items_initial():
    '''Adding a initial report item does what is expected'''
    printer = printout.Printer()
    data = ['one item', 'another']
    printer.add_report_items(MockDirobject(), data)
    assert len(printer.report_items) == len(data)


def test_printer_add_report_items_second():
    '''Adding a second report item does what is expected'''
    printer = printout.Printer()
    dir = MockDirobject()
    data = ['one item', 'another']
    printer.add_report_items(dir, data)
    data = ['third']
    printer.add_report_items(dir, data)
    assert len(printer.report_items) == 3


def test_printer_add_report_items_filtered():
    '''Stock parens are filtered out of parens added.'''
    printer = printout.Printer(set(['ignored']))
    data = ['one item', 'ignored', 'another']
    printer.add_report_items(MockDirobject(), data)
    assert printer.report_items == [(EXAMPLE_PATH, 'one item'),
                                    (EXAMPLE_PATH, 'another')]


# print_report_items()

def test_printer_print_report_items(capsys):
    '''Printing report items prints a line per item'''

    printer = printout.Printer()
    data = ['one item', 'another']
    printer.add_report_items(MockDirobject(), data)
    printer.print_report_items()
    (out, err) = capsys.readouterr()
    assert out.count('\n') == len(data)
    assert err == ''
